package sr5e1_clk.util

import java.util.Collection
import java.util.List
import java.util.Map
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.common.util.DiagnosticChain
import org.eclipse.emf.ecore.EAttribute
import org.eclipse.emf.ecore.EDataType
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.emf.ecore.util.ExtendedMetaData
import org.eclipse.emf.ecore.util.FeatureMap
import org.eclipse.emf.ecore.util.FeatureMapUtil
import sr5e1_clk.ADC_SEL_VALUE
import sr5e1_clk.Clock
import sr5e1_clk.FDCAN_SEL_VALUE
import sr5e1_clk.I2C_SEL_VALUE
import sr5e1_clk.I2S_SEL_VALUE
import sr5e1_clk.MCO_SEL_VALUE
import sr5e1_clk.MSr5e1_clkPackage
import sr5e1_clk.PLL0_SEL
import sr5e1_clk.PLL1_SEL
import sr5e1_clk.RTC_SEL_VALUE
import sr5e1_clk.SDADC_SEL_VALUE
import sr5e1_clk.SW_VALUE
import sr5e1_clk.UART_SEL_VALUE

class MySr5e1_clkValidator extends Sr5e1_clkValidator {

	public static final MySr5e1_clkValidator INSTANCE = new MySr5e1_clkValidator();

	/**
	 * The current clock to be validated.
	 * This is needed to access to other attributes when an attribute is being validated
	 */
	Clock currentClock

	override validateXOSC_VALUE_myConstraint(long XOSC, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (XOSC % 2 == 1) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, DIAGNOSTIC_SOURCE, 0, "_UI_GenericConstraint_diagnostic",
						#["myConstraint", getValueLabel(MSr5e1_clkPackage.Literals.XOSC_VALUE, XOSC, context)], #[XOSC],
						context))
			}
			return false
		}
		return true
	}

	/**
	 * Overridden method to set the Clock being validated
	 */
	override boolean validateClock(Clock clock, DiagnosticChain diagnostics, Map<Object, Object> context) {
		currentClock = clock
		return super.validateClock(clock, diagnostics, context)
	}

	/**
	 * Validation of XOSC value.
	 * Is disabled when XOSC is disabled
	 * LOW: StellarERCR #1029769: Oscillator Switches Settings not fully managed in SR5E1 clocktree
	 */
	override validateXOSC_VALUE(long xosC_VALUE, DiagnosticChain diagnostics, Map<Object, Object> context) {

		// XOSC value minimal and maximum are not to be taken into account when XOSC is not enabled
		val clock = currentClock
		if (clock !== null && clock.XOSC_Enabled) {
			super.validateXOSC_VALUE(xosC_VALUE, diagnostics, context)
		} else {
			// Validation is OK whatever the value of the XOSC value
			true
		}
	}

	override validateLSI_VALUE(long lsI_VALUE, DiagnosticChain diagnostics, Map<Object, Object> context) {
		// XOSC value minimal and maximum are not to be taken into account when XOSC is not enabled
		val clock = currentClock
		if (clock !== null && clock.XOSC_Enabled) {
			super.validateLSI_VALUE(lsI_VALUE, diagnostics, context)
		} else {
			// Validation is OK whatever the value of the XOSC value
			true
		}
	}

	override validateSDADC_VALUE(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		// XOSC value minimal and maximum are not to be taken into account when value is 0
		if (value === 0) {
			true
		} else {
			super.validateSDADC_VALUE(value, diagnostics, context)
		}
	}

	override validatePLL1_PHI(long pll1, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (pll1 === 0) {
			true
		} else {
			super.validatePLL1_PHI(pll1, diagnostics, context)
		}
	}

	override validatePLL1_IN(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL1_IN(value, diagnostics, context)
		}
	}

	override validatePLL0_IN(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL0_IN(value, diagnostics, context)
		}
	}

	override validatePLL0_PFD(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL0_PFD(value, diagnostics, context)
		}
	}

	override validatePLL0_PHI(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL0_PHI(value, diagnostics, context)
		}
	}

	override validatePLL0_PHI1(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL0_PHI1(value, diagnostics, context)
		}
	}

	override validatePLL0_VCO(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL0_VCO(value, diagnostics, context)
		}
	}

	override validatePLL1_VCO(long value, DiagnosticChain diagnostics, Map<Object, Object> context) {
		if (value === 0) {
			true
		} else {
			super.validatePLL1_VCO(value, diagnostics, context)
		}
	}

	override validatePLL0_SEL(PLL0_SEL pll0_SEL, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && !clock.XOSC_Enabled && pll0_SEL === PLL0_SEL.XOSC) {
			false
		} else {
			true
		}
	}

	override validatePLL1_SEL(PLL1_SEL pll1_SEL, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && !clock.XOSC_Enabled && pll1_SEL === PLL1_SEL.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI1 == 0 && pll1_SEL === PLL1_SEL.PLL0PHI1) {
			false
		} else {
			true
		}
	}

	override validateSW_VALUE(SW_VALUE sW_VALUE, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && sW_VALUE === SW_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.getPLL0PHI == 0 && sW_VALUE === SW_VALUE.PLL0PHI) {
			false
		} else if (clock !== null && clock.getPLL1PHI == 0 && sW_VALUE === SW_VALUE.PLL1PHI) {
			false
		} else {
			true
		}
	}

	override validateUART_SEL_VALUE(UART_SEL_VALUE uarT_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.LSI == 0 && uarT_SEL_VALUE === UART_SEL_VALUE.LSI) {
			false
		} else if (clock !== null && clock.XOSC == 0 && uarT_SEL_VALUE === UART_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && uarT_SEL_VALUE === UART_SEL_VALUE.PLL0PHI) {
			false
		} else {
			true
		}
	}

	override validateI2C_SEL_VALUE(I2C_SEL_VALUE i2C_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && i2C_SEL_VALUE === I2C_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && i2C_SEL_VALUE === I2C_SEL_VALUE.PLL0PHI) {
			false
		} else {
			true
		}
	}

	override validateI2S_SEL_VALUE(I2S_SEL_VALUE i2s_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && i2s_SEL_VALUE === I2S_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && i2s_SEL_VALUE === I2S_SEL_VALUE.PLL0PHI) {
			false
		} else if (clock !== null && clock.i2S_CKIN == 0 && i2s_SEL_VALUE === I2S_SEL_VALUE.I2S_CKIN) {
			false
		} else {
			true
		}
	}

	override validateFDCAN_SEL_VALUE(FDCAN_SEL_VALUE fdcaN_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && fdcaN_SEL_VALUE === FDCAN_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && fdcaN_SEL_VALUE === FDCAN_SEL_VALUE.PLL0PHI) {
			false
		} else {
			true
		}
	}

	override validateADC_SEL_VALUE(ADC_SEL_VALUE adC_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && adC_SEL_VALUE === ADC_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && adC_SEL_VALUE === ADC_SEL_VALUE.PLL0PHI) {
			false
		} else if (clock !== null && clock.PLL1PHI == 0 && adC_SEL_VALUE === ADC_SEL_VALUE.PLL1PHI) {
			false
		} else {
			true
		}
	}

	override validateSDADC_SEL_VALUE(SDADC_SEL_VALUE sdadc_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.XOSC == 0 && sdadc_SEL_VALUE === SDADC_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && sdadc_SEL_VALUE === SDADC_SEL_VALUE.PLL0PHI) {
			false
		} else if (clock !== null && clock.PLL1PHI == 0 && sdadc_SEL_VALUE === SDADC_SEL_VALUE.PLL1PHI) {
			false
		} else {
			true
		}
	}

	override validateMCO_SEL_VALUE(MCO_SEL_VALUE mcO_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.LSI == 0 && mcO_SEL_VALUE === MCO_SEL_VALUE.LSI) {
			false
		} else if (clock !== null && clock.XOSC == 0 && mcO_SEL_VALUE === MCO_SEL_VALUE.XOSC) {
			false
		} else if (clock !== null && clock.PLL0PHI == 0 && mcO_SEL_VALUE === MCO_SEL_VALUE.PLL0PHI) {
			false
		} else if (clock !== null && clock.PLL1PHI == 0 && mcO_SEL_VALUE === MCO_SEL_VALUE.PLL1PHI) {
			false
		} else {
			true
		}
	}

	override validateRTC_SEL_VALUE(RTC_SEL_VALUE rtC_SEL_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val clock = currentClock
		if (clock !== null && clock.LSI == 0 && rtC_SEL_VALUE === RTC_SEL_VALUE.LSI) {
			false
		} else if (clock !== null && clock.XOSC == 0 && rtC_SEL_VALUE === RTC_SEL_VALUE.XOSC) {
			false
		} else {
			true
		}
	}

	override protected validate_DataValueConforms(EObject eObject, EAttribute eAttribute, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		// Consider that default values are to be validated as well
		// if (!eObject.eIsSet(eAttribute)) {
		// return true
		// }
		var boolean result = true
		val eDataType = eAttribute.EAttributeType
		val rootValidator = getRootEValidator(context)
		val value = eObject.eGet(eAttribute)

		if (FeatureMapUtil.isFeatureMap(eAttribute)) {
			val featureMap = value as Collection<FeatureMap.Entry>
			val eClass = eObject.eClass
			var Map<EStructuralFeature, DiagnosticChain> entryFeatureToDiagnosticChainMap = null

			for (entry : featureMap) {
				if (result || diagnostics === null) {

					val entryFeature = entry.EStructuralFeature
					if (entryFeature instanceof EAttribute &&
						ExtendedMetaData.INSTANCE.getAffiliation(eClass, entryFeature) == eAttribute) {
						val entryType = entryFeature.EType as EDataType
						val entryValue = entry.value
						val entryIsValid = rootValidator.validate(entryType, entryValue, null, context)

						if (!entryIsValid) {
							result = false
							if (diagnostics !== null) {
								if (entryFeatureToDiagnosticChainMap === null) {
									entryFeatureToDiagnosticChainMap = newHashMap
								}
								var entryFeatureDiagnostic = entryFeatureToDiagnosticChainMap.get(entryFeature)
								if (entryFeatureDiagnostic === null) {
									entryFeatureDiagnostic = createBadDataValueDiagnostic(eObject,
										entryFeature as EAttribute, diagnostics, context)
									entryFeatureToDiagnosticChainMap.put(entryFeature, entryFeatureDiagnostic)
								}
								rootValidator.validate(entryType, entryValue, entryFeatureDiagnostic, context)
							}
						}
					}
				}
			}
		} else if (eAttribute.many) {
			for (item : value as List<?>) {
				if (result || diagnostics === null) {
					result = result && rootValidator.validate(eDataType, item, null, context)
				}
			}

			if (!result && diagnostics !== null) {
				val diagnostic = createBadDataValueDiagnostic(eObject, eAttribute, diagnostics, context)
				for (item : value as List<?>) {
					rootValidator.validate(eDataType, item, diagnostic, context)
				}
			}
		} else if (value !== null) {
//			result = rootValidator.validate(eDataType, value, diagnostics, context)
			result = rootValidator.validate(eDataType, value, null, context)
			if (!result && diagnostics !== null) {
				val diagnostic = createBadDataValueDiagnostic(eObject, eAttribute, diagnostics, context)
				rootValidator.validate(eDataType, value, diagnostic, context)
			}
		}

		return result
	}
}
