package com.st.stellar.component.sr5e1_can.validation

import com.st.stellar.component.API.ComponentAPI
import com.st.stellar.component.API.Utils
import com.st.stellar.component.sr5e1_can.Configuration
import com.st.stellar.component.sr5e1_can.MSr5e1_canPackage
import com.st.stellar.component.sr5e1_can.Sr5e1_can
import com.st.stellar.component.sr5e1_can.Sr5e1_canPackage
import com.st.stellar.component.sr5e1_can.impl.MConfigurationImpl
import com.st.stellar.component.sr5e1_can.impl.Sr5e1_canImpl
import com.st.stellar.component.sr5e1_can.irq_mask_t
import com.st.stellar.component.sr5e1_can.ts_mode_t
import com.st.stellar.component.sr5e1_can.util.Sr5e1_canValidator
import java.io.File
import java.util.HashSet
import java.util.Map
import java.util.regex.Pattern
import org.eclipse.core.resources.IProject
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.core.runtime.Path
import org.eclipse.emf.common.util.BasicDiagnostic
import org.eclipse.emf.common.util.Diagnostic
import org.eclipse.emf.common.util.DiagnosticChain
import org.eclipse.emf.ecore.EAttribute
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emf.common.util.URI
import com.st.stellar.component.Component

class MySr5e1_canValidator extends Sr5e1_canValidator {

	public static final MySr5e1_canValidator INSTANCE = new MySr5e1_canValidator();

	Sr5e1_can currentComponent = null
	Configuration currentConfiguration = null

	static val ISSUE_CODE_PREFIX = "com.st.stellar.component.sr5e1_can."

	static val WRONG_NAME = ISSUE_CODE_PREFIX + 'wrongName'

	static val WRONG_HEX = ISSUE_CODE_PREFIX + 'wrongHexadecimal'

	static val WRONG_CB = ISSUE_CODE_PREFIX + 'wrongCallback'

	override validateConfiguration(Configuration configuration, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		currentComponent = configuration.eContainer as Sr5e1_can
		currentConfiguration = configuration
		super.validateConfiguration(configuration, diagnostics, context)
	}

	override validateSr5e1_can(Sr5e1_can comp, DiagnosticChain diagnostics, Map<Object, Object> context) {
		currentComponent = comp
		val res = super.validateSr5e1_can(comp, diagnostics, context)
		return res
	}

	override validatename_t_checkName(String nameToValidate, DiagnosticChain diagnostics, Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		var res = checkName(nameToValidate, diagnostics, context)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_NAME, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
						#["checkName", getValueLabel(MSr5e1_canPackage.Literals.NAME_T, nameToValidate, context)],
						#[nameToValidate], context))
				res = false
			}
		}
		return res
	}

	static val WRONG_PATH = ISSUE_CODE_PREFIX + 'wrongPath'

	override validatePath_checkPath(String path, DiagnosticChain diagnostics, Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		var res = checkInputPath(path, diagnostics, context)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_PATH, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
						#["checkPath", getValueLabel(MSr5e1_canPackage.Literals.PATH, path, context)], #[path],
						context))
				res = false
			}
		}
		return res
	}

	def checkInputPath(String location, DiagnosticChain chain, Map<Object, Object> map) {
		val comp = currentComponent
		if(comp === null) return true

		val project = getProject(comp)
		val fileName = project.getLocation().toOSString + File.separator + location
		val path = new Path(fileName)
		var res = false
		if (path.isValidPath(fileName)) {
			val file = new File(fileName)
			res = file.exists
		}
		res
	}

	public static val WRONG_GENPATH = ISSUE_CODE_PREFIX + 'wrongGenPath'

	override validategenpath_t_checkGenPath(String path, DiagnosticChain diagnostics, Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		var res = checkPath(path, diagnostics, context)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_GENPATH, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
						#["checkPath", getValueLabel(MSr5e1_canPackage.Literals.GENPATH_T, path, context)], #[path],
						context))
				res = false
			}
		}
		return res
	}

	override validateuint32_t_checkUINT32_T(Integer uint32_t, DiagnosticChain diagnostics,
		Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		if (uint32_t >= 0 && uint32_t <= 4294967295L) {
			return true
		} else {
			return false
		}
	}

	override validatedbrp_t_checkDbrp_t(Integer dbrp_t, DiagnosticChain diagnostics, Map<Object, Object> context) {
		val parent = currentComponent as Sr5e1_canImpl
		val config = currentConfiguration
		if (parent === null) {
			return true
		}

		if (config.delay_compensation) {
			if (dbrp_t >= 1 && dbrp_t <= 2) {
				return true
			} else {
				return false
			}
		} else {
			if (dbrp_t >= 1 && dbrp_t <= 32) {
				return true
			} else {
				return false
			}
		}
	}

	override validatets_presc_t_checkTs_presc_t(Integer ts_presc_t, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		val parent = currentComponent as Sr5e1_canImpl
		val config = currentConfiguration
		if (parent === null) {
			return true
		}

		val src = config.timestamp_source;

		if (src == ts_mode_t.CAN_TIMESTAMP_MODE_INT) {
			if (ts_presc_t >= 1 && ts_presc_t <= 16) {
				return true
			} else {
				return false
			}
		}
		if (src == ts_mode_t.CAN_TIMESTAMP_MODE_EXT || src == ts_mode_t.CAN_TIMESTAMP_MODE_TSU) {
			if (ts_presc_t >= 1 && ts_presc_t <= 256) {
				return true
			} else {
				return false
			}
		}
		return true
	}

	override validateirq_mask_t_checkIRQMask(irq_mask_t irq_mask_t, DiagnosticChain diagnostics,
		Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		val config = currentConfiguration

		var counter = 0

		if (parent === null) {
			return true
		}

		val irqs = config.custom_IRQ_mask.map(ch|ch.literal)

		if (irqs === null) {
			return true
		}

		for (irq : irqs) {
			if (irq == irq_mask_t.literal) {
				counter++
			}
		}

		if (counter >= 2) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, DIAGNOSTIC_SOURCE, CONFIG_CODE,
						"_UI_GenericConstraint_diagnostic",
						#["checkIRQMask", getValueLabel(MSr5e1_canPackage.Literals.GENPATH_T, irq_mask_t, context)],
						#[irq_mask_t], context))
			}
			return false
		} else {
			return true
		}
	}

	override validatehex_checkHex(String hex, DiagnosticChain diagnostics, Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		var res = checkHex(hex, diagnostics, context)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_HEX, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
						#["checkHex", getValueLabel(MSr5e1_canPackage.Literals.GENPATH_T, hex, context)], #[hex],
						context))
				res = false
			}
		}
		return res
	}

	override validatecallback_t_checkCb(String callback_t, DiagnosticChain diagnostics, Map<Object, Object> context) {

		val parent = currentComponent as Sr5e1_canImpl
		var res = true

		if (parent === null) {
			return true
		}

		var eClass = currentConfiguration.eClass
		var callbackList = eClass.getEAllStructuralFeatures.filter(EAttribute).filter [
			EAttributeType.equals(Sr5e1_canPackage.eINSTANCE.getcallback_t)
		].map [ attr |
			{
				val nameValue = currentConfiguration.eGet(attr)
				val value = nameValue !== null ? EcoreUtil.convertToString(attr.getEAttributeType(), nameValue) : ""
				value
			}
		].filterNull

		if (callbackList !== null && !callbackList.isEmpty) {
			// The new callback name must not be a duplicate
			val thereAreDuplicates = (callbackList.filter[it.equals(callback_t) && callback_t.length > 0].size > 1)
			if (thereAreDuplicates) {
				if (diagnostics !== null) {
					diagnostics.add(
						createDiagnostic(Diagnostic.ERROR, WRONG_CB, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
							#["duplicated callback name",
								getValueLabel(MSr5e1_canPackage.Literals.CALLBACK_T, callback_t, context)],
							#[callback_t], context))
				}
				res = false
				return res
			}
		}
		res = checkCb(callback_t, diagnostics, context)
		if (!res) {
			if (diagnostics !== null) {
				diagnostics.add(
					createDiagnostic(Diagnostic.ERROR, WRONG_CB, CONFIG_CODE, "_UI_GenericConstraint_diagnostic",
						#["checkPath", getValueLabel(MSr5e1_canPackage.Literals.CALLBACK_T, callback_t, context)],
						#[callback_t], context))
				res = false
			}
		}
		return res
	}

	static val REGEX_HEX = Pattern.compile('''^0x[0-9a-fA-F]{8}$''')

	def checkHex(String hex, DiagnosticChain chain, Map<Object, Object> map) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		val matches = REGEX_HEX.matcher(hex).matches
		if (!matches) {
			return false
		} else {
			return true
		}
	}

	static val REGEX_NAME = Pattern.compile('''^[a-zA-Z]+[a-zA-Z0-9_]*$''')

	def checkName(String nameToValidate, DiagnosticChain chain, Map<Object, Object> map) {

		val parent = currentComponent as Sr5e1_canImpl
		if (parent === null) {
			return true
		}

		val matches = REGEX_NAME.matcher(nameToValidate).matches
		if (!matches) {
			return false
		}

		// The name is syntactically correct... but may be already present in other configurations
		// Create the list of configuration names already present inside the Sr5e1_can main object being validated
		// filter out current configuration (with its old name)
		var name_list = parent.configs.filter(c|c !== currentConfiguration).map[it|(it as MConfigurationImpl).name].
			toList
		if (name_list === null) {
			return true
		}

		// The new name must not be a duplicate
		val noDuplicates = (name_list.filter[it.equals(nameToValidate)].size < 1)
		noDuplicates
	}

	static val REGEX_CB = Pattern.compile('''^([a-zA-Z][a-zA-Z0-9_-]*)?$''')

	def checkCb(String location, DiagnosticChain chain, Map<Object, Object> map) {
		val comp = currentComponent
		if(comp === null) return true
		val matcher = REGEX_CB.matcher(location)
		var res = matcher.matches()
		res
	}

	static val REGEX_PATH = Pattern.compile('''^[a-zA-Z0-9-_]+[a-zA-Z0-9-/\\_]*$''')

	def checkPath(String location, DiagnosticChain chain, Map<Object, Object> map) {
		val comp = currentComponent
		if(comp === null) return true
		val matcher = REGEX_PATH.matcher(location)
		var res = matcher.matches()
		res
	}

	static def IProject getProject(EObject model) {
		val uri = model.eResource.URI
		val projectName = uri.segmentsList().get(1).toString()
		return ResourcesPlugin.workspace.root.getProject(projectName)
	}

	static val MISSING_EXTERNAL_LINK = ISSUE_CODE_PREFIX + 'missingExternalLink'

	static val int CAN_CODE = 98 as int
	static val int CONFIG_CODE = 99 as int

	override validateSr5e1_can_checkExternalLinks(Sr5e1_can sr5e1_can, DiagnosticChain diagnostics,
		Map<Object, Object> context) {

		var res = true
		if(currentComponent.eResource() === null) return true
		val app = ComponentAPI.getApplication(currentComponent.eResource)
		val file = Utils.getFileFromResource(currentComponent.eResource().getURI());
		val project = file.getProject();

		val expectedConfigurators = currentComponent.requiredFeatures

		val missingFeatures = new HashSet<String>()

		for (expectedConfigurator : Utils.removeDuplicates(expectedConfigurators)) {
			var configuratorsList = newHashSet
			for (ref : app.components.filter[isEnabled]) {
				val uriToValidateStr = "/" + project.getName() + "/" + expectedConfigurator
				val uriToValidate = URI.createPlatformResourceURI(uriToValidateStr, true);
				if (ComponentAPI.isEnabled(uriToValidate.toString)) {
					configuratorsList.add(ref.toString)
				}
			}
			if( configuratorsList.isEmpty) {
				missingFeatures.add(expectedConfigurator)
			}
		}
		if (!missingFeatures.isEmpty) {
			if (diagnostics !== null) {
				for (configurator : missingFeatures) {
					diagnostics.add(
						new BasicDiagnostic(Diagnostic.ERROR, MISSING_EXTERNAL_LINK,
							CAN_CODE, '''Missing configurator '«configurator»' in application.''', #[sr5e1_can]))
				}
			}
			res = false
		}
		return res
	}

	override validateFDCAN_VALUE_checkExternalLinks(long fdcaN_VALUE, DiagnosticChain diagnostics,
		Map<Object, Object> context) {
		return true
	}

}
